<?PHP if ( ! defined('BASEPATH')) exit('No direct script access allowed');
/**
* @package dpii
* @subpackage models
*/

/** require parent */
require_once 'database_model.php';

/**
* @package dpii
* @subpackage models
*/
class User_model extends Database_model {
	protected $primary_key = 'user_id';
	protected $_logged_in_user; //set in is_logged_in()
	protected $_is_admin;
	
	//////////////////////////////////////////
	// METHODS CONCERNING THE LOGGED IN USER
	//////////////////////////////////////////
	
	/**
	* True if a user is logged in.
	* @todo Allow a username to be passed in to check if a specific user is logged in
	* @return boolean
	*/
	public function is_logged_in(){
		if($this->session->userdata("is_loggedin") != "true") 
			return false;
		$username = $this->session->userdata('username');
		if(empty($username)) return false;
		
		//if this is the first time we've called this method since the page load, make sure that the user exists & store them for future reference.
		if(!isset($this->_logged_in_user) || $this->_logged_in_user['user_name'] != $username){
			$user = $this->db->limit(1)->get_where('users', array('user_name' => $username, 'user_deleted_flag' => 0))->row_array();
			if(empty($user) || !is_array($user)) return false;
			$this->_logged_in_user = $user;
			$this->_logged_in_user['dn'] = "uid=" . $username . "," . LDAP_ACCOUNTS_DN;
		}
		
		return true;
	}
	
	/**
	* True if the logged in user is an admin.
	* @todo Allow a username to be passed in ot check if a specific user is an admin.
	* @return boolean
	*/
	public function is_admin(){
		if(!$this->is_logged_in()) return false;
		if(!isset($this->_is_admin)){
			$this->load->library('ldap');
			$groups = $this->ldap->get_admin_group_membership($this->logged_in_user('dn'));
			$this->_is_admin = $this->is->nonzero_unsigned_integer($groups['count']);
		}
		return $this->_is_admin;
	}
	
	/**
	* True if the user user has been active within the time allowed by {@link SESSION_TIMEOUT_MINS}.
	* Additionally, updates the last activity count if the user has not been timed out.
	* @return boolean
	*/
	public function has_recent_activity(){
		//prev last activity update set to current last activity so we can reset it for functions we don't want to consider "activity"
		if(strlen($this->session->userdata('app_last_activity')) > 0) { 
			$this->session->set_userdata('prev_last_activity',$this->session->userdata('app_last_activity')); 
		}else{ //if last activity not yet set, set it to current time
			$this->session->set_userdata('app_last_activity',time()); 
		}
		
        //check if it the user needs to be logged out due to inactivity
        if(abs($this->session->userdata('app_last_activity') - time()) > (SESSION_TIMEOUT_MINS*60)){ 
			return false;
		}

		//update the last activity for the timeout
		$this->session->set_userdata('app_last_activity',time()); 
		return true;
	}
	
	/**
	* The credentials that should be used to sign into LDAP.
	* Generally, this is the logged-in user's own credentials.  If no one is logged in,
	* defaults to the credentials for the {@link LDAP_SEARCH_USERNAME} account.
	* @return array
	*/
	public function ldap_credentials(){
		$username = $this->logged_in_user('user_name');
		if(!empty($username)){
			return array( 'user' => $username,
						  'pwd' => $this->encrypt->decode($this->session->userdata('ep')));
		}
		return array('user' => LDAP_SEARCH_USERNAME, 'pwd' => LDAP_SEARCH_PASSWORD);
	}
	
	/**
	* 
	*
	*/
	public function logged_in_user($field = null){				
		if(!$this->is_logged_in()) return array();
		if(is_null($field))
			return $this->_logged_in_user;
		if(!array_key_exists($field, $this->_logged_in_user)) return $this->error->should_be_a_recognized_user_property($field);
		return $this->_logged_in_user[$field];		
	}
	
	public function log_out($destination = 'auth'){
		$this->session->sess_destroy(); 
		session_destroy(); 
		redirect($destination);
	}
	
	////////////////////////////////////////
	// METHODS CONCERNING ALL USERS
	////////////////////////////////////////
	
	public function username_from_id($id) {
		if(!$this->is->nonzero_unsigned_integer($id)) return $this->error->should_be_a_nonzero_unsigned_int($id);
		$user = $this->db->limit(1)->get_where('users', array('user_id' => $id));
		if($user && $user->num_rows > 0) { 
			$result = $user->row_array(0);
			return $result['user_name'];
		}
		return FALSE;
	}
	
	public function user_cn_from_id($id) {
		if(!$this->is->nonzero_unsigned_integer($id)) return $this->error->should_be_a_nonzero_unsigned_int($id);
		$result = $this->ldap->search($this->username_from_id($id),1,array('cn'));
		if(count($result) > 0) { return $result[0]['cn']; }
		return FALSE;
	}
}